#ifndef ANIMATION_H
#define ANIMATION_H

#pragma warning(disable:4786)
#include <set>
#include <algorithm>

using namespace std;

#include "rwcommon.h"
#include "rtquat.h"
#include "rtslerp.h"
#include "rpskin.h"
#include "rphanim.h"

#define SMALL_FLOAT 0.00001

enum animCurveType
{
    translationAnimCurve = 0,
    rotationAnimCurve,
    scaleAnimCurve,
    unknownAnimCurve
};

/* Class representing a single animation key. */
class animKey
{
public:
    bool        isKeyFrame;
    float       time;
    bool        dirty;
    RwMatrix    matrix;
    RpQuat      quat;
    RwInt32     nodeIndex;

    animKey(float time);
    animKey(float time, bool isKeyFrame);
    animKey(float time, RwMatrix *matrix, bool isKeyFrame);

    bool operator==(const animKey &other) const;
    bool operator<(const animKey &other) const;
    animKey & operator*=(animKey &b);
};

typedef set<animKey> animKeys;
typedef set<animKey>::iterator animKeyIt;

/*
    Class representing a list of animation keys. We
    Use a set to ensure that the list is sorted and
    does not contain duplicates.
*/
class animKeyList
{
public:
    animKeys    keys;

    animKeyList & operator*=(animKeyList &b);

    RwReal      getKeyRange();
    void        copyKeyFrames(animKeyList &src, animKeyList &dst,
                                float offset, bool reverse);
    void        dumpKeyFrames();
    bool        hasRotationAnimation();
    bool        hasTranslationAnimation();
    void        checkKeyFrameErrors(float errorTolerance);
    void        computeAnimation(int mdtIndex, MDagPath dagPath);
    void        removeRedundantKeys();
    void        convertToRpAnim(RwFrame *frame, RwChar *animName);
    void        addToHierarchyKeyFrames(animKeys &hierarchyKeys, unsigned int nodeIndex);
    int         countValidKeys();

private:

    RwV3d       posRange;
    RpQuat      quatRange;

    void        checkKeyFrameRangeError(animKeyIt startKey, animKeyIt endKey,
                                int rangeKeyCount, float errorTolerance);
    bool        checkKeyValueErrors(animKeyIt startKey, animKeyIt middleKey,
                                animKeyIt endKey, RwReal errorTolerance);
    bool        checkKeyValueError(float startTime, float endTime,
                                float startValue, float endValue,
                                float interpEndValue, float errorTolerance,
                                float keyrange);
    inline void rangeCheck(float &min, float &max, float value);
    void        computeKeyFrameRange();
    void        addAnimKeysFromTransform(MObject transform);
    void        addKeysFromParents(MFnDagNode &nodeDagFn);
    void        addAnimKeysFromAnimCurve(MObject anim, MObject transform);
};

/* General animation functions */
animCurveType   getAnimCurveType(MObject animCurve, MObject object);
void            generateAnimFromHierarchyKeyFrames(animKeys &hierarchyKeys,
                            RpSkinAnim **skinAnim, RpHAnimAnimation **hAnim);

#endif //ANIMATION_H